//
// Created by niclas on 7/8/25.
//

#include <max14906.h>
#include <string.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "main.h"
#include "commands.h"
#include "ina238.h"
#include "jump-bootloader.h"

extern max14906_handle_t max14906_da;
extern max14906_handle_t max14906_ds;

extern ina238_handle_t ina238_in;
extern ina238_handle_t ina238_rcmp;
extern ina238_handle_t ina238_payment;
extern ina238_handle_t ina238_screen;
extern ina238_handle_t ina238_router;
extern mcp4725_handle_t dac;

extern UART_HandleTypeDef huart1;
extern UART_HandleTypeDef huart3;
extern SPI_HandleTypeDef hspi2;
extern UART_HandleTypeDef *uart;
extern SPI_HandleTypeDef *spi;

extern TIM_HandleTypeDef *htim_pwm;
extern I2C_HandleTypeDef *hi2c;

extern uint64_t hu_counter;
extern uint64_t hv_counter;
extern uint64_t hw_counter;

static char conversion[40];

static HAL_StatusTypeDef status;

// Forward declarations of command handlers
static void help_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void bootloader_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs);
static void da_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void out_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void fr_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void sv_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void read_ds_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void read_hall_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void read_port_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void rcmp_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void rcmp_voltage_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void rcmp_current_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void rcmp_power_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void rcmp_alert_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg);
static void max14906_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs);
static void measure_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs);
//static void i2c_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs);

cmd_mapping commands[] = {
    {"help", help_handler},
    {"bootmode", bootloader_handler},
    {"da1_on", da_handler, 1},
    {"da2_on", da_handler, 2},
    {"da3_on", da_handler, 3},
    {"da1_off", da_handler, 17},
    {"da2_off", da_handler, 18},
    {"da3_off", da_handler, 19},
    {"out1_on", out_handler, 1},
    {"out2_on", out_handler, 2},
    {"out3_on", out_handler, 3},
    {"out4_on", out_handler, 4},
    {"out1_off", out_handler, 17},
    {"out2_off", out_handler, 18},
    {"out3_off", out_handler, 19},
    {"out3_off", out_handler, 20},
    {"ds1", read_ds_handler, 1},
    {"ds2", read_ds_handler, 2},
    {"ds3", read_ds_handler, 3},
    {"fr_on", fr_handler, 0},
    {"fr_off", fr_handler, 1},
    {"sv", sv_handler, -1},
    {"rcmp_on", rcmp_handler, 0},
    {"rcmp_off", rcmp_handler, 1},
    {"rcmp_voltage", rcmp_voltage_handler, 0},
    {"rcmp_current", rcmp_current_handler, 0},
    {"rcmp_power", rcmp_power_handler, 0},
    {"rcmp_alert", rcmp_alert_handler, 0},
    {"hu", read_hall_handler, 1},
    {"hu_count", read_hall_handler, 17},
    {"hv", read_hall_handler, 2},
    {"hv_count", read_hall_handler, 18},
    {"hw", read_hall_handler, 3},
    {"hw_count", read_hall_handler, 19},
    {"porta", read_port_handler, 1},
    {"portb", read_port_handler, 2},
    {"portc", read_port_handler, 3},
    {"portd", read_port_handler, 4},
//    {"i2c", i2c_handler, 0},
	{"measure1", measure_handler, 1},
    {"max_ds", max14906_handler, 1},
    {"max_da", max14906_handler, 2}
};

// Serial comms
void comm_format_append(char *output, char const *text, uint32_t const value, bool const wide) {
    char tmp[40];
    strcpy(tmp, text);
    strcat(tmp, "%");
    if (wide) {
        strcat(tmp, "4l");
    }
    strcat(tmp, "x\n");
    snprintf(conversion, sizeof(conversion), tmp, value);
    strcat(output, conversion);
}

int checksum(char *text) {
    int cks = 0;
    for (int i = 0; i < strlen(text); i++) {
        cks = cks << 2;
        cks = cks ^ text[i];
        cks = cks << 5;
        cks = cks + text[i];
    }
    return cks;
}

bool starts_with(const char *str, const char *prefix) {
    return strncmp(str, prefix, strlen(prefix)) == 0;
}

char *find_args(char *cmd) {
    char *found = strchr(cmd, ' ');
    if(found == NULL)
    	return "";
    while( found[0] == ' ') {  // trim starting spaces
    	found = &found[1];
    }
    return found;
}

void cmd_execute(char *cmd, char *response, uint16_t maxlen) {
	if( strlen(cmd) == 0) {
		return;
	}
    response[0] = '\0';
    bool found = false;
    for (size_t i = 0; i < sizeof(commands) / sizeof(commands[0]); i++) {
        if (starts_with(cmd, commands[i].cmd_str)) {
            uint32_t arg = commands[i].arg;
            cmd = find_args(cmd);
            commands[i].cmd_func(cmd, response, maxlen, arg);
            found = true;
        }
    }
    // response
    if (!found)
        strcpy(response, "No such command.");
    uint16_t cks = checksum(response);
    sprintf(conversion, "\r\n .ok(%d) ", status);
    strcat(response, conversion);
    sprintf(conversion, "[%4x]\r\n", cks);
    strcat(response, conversion);
    HAL_UART_Transmit_DMA(uart, (uint8_t *) response, strlen(response));
}

// Command handler implementations
static void help_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    strcpy(response, "Here are the commands:\n");
    for (int i = 0; i < sizeof(commands) / sizeof(commands[0]); i++) {
        strcat(response, "  ");
        strcat(response, commands[i].cmd_str);
        strcat(response, "\n");
    }
}

static void set_da(uint8_t pin_number) {
	switch(pin_number){
	case 1:
		HAL_GPIO_WritePin(M_DA1_GPIO_Port, M_DA1_Pin, GPIO_PIN_SET);
		break;
	case 2:
		HAL_GPIO_WritePin(M_DA2_GPIO_Port, M_DA2_Pin, GPIO_PIN_SET);
		break;
	case 3:
		HAL_GPIO_WritePin(M_DA3_GPIO_Port, M_DA3_Pin, GPIO_PIN_SET);
		break;
	}
}

static void reset_da(uint8_t pin_number) {
	switch(pin_number){
	case 1:
		HAL_GPIO_WritePin(M_DA1_GPIO_Port, M_DA1_Pin, GPIO_PIN_RESET);
		break;
	case 2:
		HAL_GPIO_WritePin(M_DA2_GPIO_Port, M_DA2_Pin, GPIO_PIN_RESET);
		break;
	case 3:
		HAL_GPIO_WritePin(M_DA3_GPIO_Port, M_DA3_Pin, GPIO_PIN_RESET);
		break;
	}
}

static void da_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    if (arg < 16)
        set_da(arg);
    else
        reset_da(arg & 0x0f);
}

static void fr_handler(char *cmd, char * response, uint16_t maxlen, uint32_t arg ) {
    HAL_GPIO_WritePin(FR_STM_GPIO_Port, FR_STM_Pin, arg == 0 ? GPIO_PIN_RESET : GPIO_PIN_SET);
}

static void sv_handler(char *cmd, char * response, uint16_t maxlen, uint32_t arg ) {
	if( strlen(cmd) < 1)
		return;
	double value = atof(cmd) / 100.0;  // Get argument, scaled 0.0 to 1.0
	if( value > 0.5){
		value = 0.5;	// Limit to max 5V
	}

	value = value * 0.62;  // Scale to compensate for OpAmp amplification.

	uint16_t dac_value = (uint16_t) (value * 4096.0);
	if( dac_value > 4095)
		dac_value = 4095;
	mcp4725_set_output(&dac, dac_value);

	uint16_t pwm = (uint16_t) round(value * 500);
	if( pwm > 499 )
	{
		pwm = 499;
	}
	__HAL_TIM_SET_COMPARE(htim_pwm, TIM_CHANNEL_3, pwm);

}

static void out_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    if (arg < 16)
        // SET
        switch (arg) {
            case 1:
                HAL_GPIO_WritePin(OUT1_GPIO_Port, OUT1_Pin, GPIO_PIN_SET);
                break;
            case 2:
                HAL_GPIO_WritePin(OUT2_GPIO_Port, OUT2_Pin, GPIO_PIN_SET);
                break;
            case 3:
                HAL_GPIO_WritePin(OUT3_GPIO_Port, OUT3_Pin, GPIO_PIN_SET);
                break;
            case 4:
                HAL_GPIO_WritePin(OUT4_GPIO_Port, OUT4_Pin, GPIO_PIN_SET);
                break;
            default:
                break;
        }
    else
        // RESET
        switch (arg & 0x0F) {
            case 1:
                HAL_GPIO_WritePin(OUT1_GPIO_Port, OUT1_Pin, GPIO_PIN_RESET);
                break;
            case 2:
                HAL_GPIO_WritePin(OUT2_GPIO_Port, OUT2_Pin, GPIO_PIN_RESET);
                break;
            case 3:
                HAL_GPIO_WritePin(OUT3_GPIO_Port, OUT3_Pin, GPIO_PIN_RESET);
                break;
            case 4:
                HAL_GPIO_WritePin(OUT4_GPIO_Port, OUT4_Pin, GPIO_PIN_RESET);
                break;
            default:
                break;
        }
}

static void read_ds_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    GPIO_PinState pin_state = GPIO_PIN_RESET;
    switch (arg) {
        case 1:
            pin_state = HAL_GPIO_ReadPin(DS1_GPIO_Port, DS1_Pin);
            break;
        case 2:
            pin_state = HAL_GPIO_ReadPin(DS2_GPIO_Port, DS2_Pin);
            break;
        case 3:
            pin_state = HAL_GPIO_ReadPin(DS3_GPIO_Port, DS3_Pin);
            break;
        default:
            break;
    }
    response[0] = pin_state == GPIO_PIN_SET ? '1' : '0';
    response[1] = '\0';
}

static void read_hall_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    if (arg < 16) {
        // Read state
        GPIO_PinState pin_state = GPIO_PIN_RESET;
        switch (arg) {
            case 1:
                pin_state = HAL_GPIO_ReadPin(HU_GPIO_Port, HU_Pin);
                break;
            case 2:
                pin_state = HAL_GPIO_ReadPin(HV_GPIO_Port, HV_Pin);
                break;
            case 3:
                pin_state = HAL_GPIO_ReadPin(HW_GPIO_Port, HW_Pin);
                break;
            default:
                break;
        }
        response[0] = pin_state == GPIO_PIN_SET ? '1' : '0';
        response[1] = '\0';
    } else {
        uint64_t value = 0;
        switch (arg & 0x0F) {
            case 1:
                value = hu_counter;
                break;
            case 2:
                value = hv_counter;
                break;
            case 3:
                value = hw_counter;
                break;
            default:
                break;
        }
        snprintf(response, maxlen, "%llu", value);
    }
}

static void read_port_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    GPIO_TypeDef *port = NULL;
    switch (arg) {
        case 1:
            strcat(response, "\nPortA\n");
            port = GPIOA;
            break;
        case 2:
            strcat(response, "\nPortB\n");
            port = GPIOB;
            break;
        case 3:
            strcat(response, "\nPortC\n");
            port = GPIOC;
            break;
        case 4:
            strcat(response, "\nPortD\n");
            port = GPIOD;
            break;
        default:
            break;
    }
    if (port != NULL) {
        comm_format_append(response, "  MODER: ", port->MODER, true);
        comm_format_append(response, " OTYPER: ", port->OTYPER, true);
        comm_format_append(response, "OSPEEDR: ", port->OSPEEDR, true);
        comm_format_append(response, "  PUPDR: ", port->PUPDR, true);
        comm_format_append(response, "    IDR: ", port->IDR, true);
        comm_format_append(response, "    ODR: ", port->ODR, true);
        comm_format_append(response, "   BSRR: ", port->BSRR, true);
        comm_format_append(response, "    BRR: ", port->BRR, true);
        comm_format_append(response, "   LCKR: ", port->LCKR, true);
        comm_format_append(response, "  HSLVR: ", port->HSLVR, true);
        comm_format_append(response, "SECCFGR: ", port->SECCFGR, true);
        comm_format_append(response, " AFR[0]: ", port->AFR[0], true);
        comm_format_append(response, " AFR[1]: ", port->AFR[1], true);
        comm_format_append(response, " AFR[2]: ", port->AFR[2], true);

    }
}

static void rcmp_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    if (arg == 0)
        HAL_GPIO_WritePin(RCMP_STM_GPIO_Port, RCMP_STM_Pin, GPIO_PIN_SET);
    if (arg == 1)
        HAL_GPIO_WritePin(RCMP_STM_GPIO_Port, RCMP_STM_Pin, GPIO_PIN_RESET);
}

static void rcmp_voltage_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    ina238_get_bus_voltage_measurement(&ina238_rcmp);
    sprintf(response, "%ld", ina238_rcmp.v32);
}

static void rcmp_current_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    ina238_get_current_result(&ina238_rcmp);
    sprintf(response, "%ld", ina238_rcmp.v32);
}

static void rcmp_power_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {
    ina238_get_power_result(&ina238_rcmp);
    sprintf(response, "%ld", ina238_rcmp.v32);
}

static void rcmp_alert_handler(char *cmd, char *response, uint16_t maxlen, uint32_t arg) {

    GPIO_PinState pin_state = HAL_GPIO_ReadPin(PWR_ALERT_GPIO_Port, PWR_ALERT_Pin);
    response[0] = pin_state == GPIO_PIN_SET ? '1' : '0';
    response[1] = '\0';
}

static uint16_t spi_read(max14906_handle_t *max, uint8_t reg) {
	uint16_t output;
	max14906_read(max, reg, &output);
	return output;
}

static void max14906_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs) {
	max14906_handle_t *max;
    if( cs == 1)
    	max = &max14906_ds;
    else if( cs == 2)
    	max = &max14906_da;
    else
    	return;

    comm_format_append(response, "SPICS", cs, false);
    comm_format_append(response, "     SetOUT: ", spi_read(max, 0x00), false);
    comm_format_append(response, "     SetLED: ", spi_read(max, 0x01), false);
    comm_format_append(response, "   DoiLevel: ", spi_read(max, 0x02), false);
    comm_format_append(response, "  Interrupt: ", spi_read(max, 0x03), false);
    comm_format_append(response, "   OvrLdChF: ", spi_read(max, 0x04), false);
    comm_format_append(response, "  OpnWirChF: ", spi_read(max, 0x05), false);
    comm_format_append(response, "  ShtVDDChF: ", spi_read(max, 0x06), false);
    comm_format_append(response, "  GlobalErr: ", spi_read(max, 0x07), false);
    comm_format_append(response, "   OpenWrEn: ", spi_read(max, 0x08), false);
    comm_format_append(response, "   ShtVDDEn: ", spi_read(max, 0x09), false);
    comm_format_append(response, "    Config1: ", spi_read(max, 0x0A), false);
    comm_format_append(response, "    Config2: ", spi_read(max, 0x0B), false);
    comm_format_append(response, "   ConfigDI: ", spi_read(max, 0x0C), false);
    comm_format_append(response, "   ConfigDO: ", spi_read(max, 0x0D), false);
    comm_format_append(response, "    CurrLim: ", spi_read(max, 0x0E), false);
    comm_format_append(response, "       Mask: ", spi_read(max, 0x0F), false);
}

static void bootloader_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs) {
    // TODO: Perhaps require a MAGIC NUMBER to be sent along as an argument

    jump_to_bootloader();
}


static void measure_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs) {
	strcpy(response, "INA238 RCMP\n");
	ina238_get_manufacturer_id(&ina238_rcmp);
    comm_format_append(response, " Manufacturer: ", ina238_rcmp.v32, true);
	ina238_get_device_id(&ina238_rcmp);
    comm_format_append(response, "    Device ID: ", ina238_rcmp.v32, true);

	ina238_get_shunt_voltage_measurement(&ina238_rcmp);
    comm_format_append(response, "Shunt Voltage: ", ina238_rcmp.v32, true);
	ina238_get_bus_voltage_measurement(&ina238_rcmp);
    comm_format_append(response, "  Bus Voltage: ", ina238_rcmp.v32, true);
	ina238_get_temperature_measurement(&ina238_rcmp);
    comm_format_append(response, "  Temperature: ", ina238_rcmp.v32, true);
	ina238_get_current_result(&ina238_rcmp);
    comm_format_append(response, "      Current: ", ina238_rcmp.v32, true);
	ina238_get_power_result(&ina238_rcmp);
    comm_format_append(response, "        Power: ", ina238_rcmp.v32, true);
	ina238_get_diagnostics(&ina238_rcmp);
    comm_format_append(response, "  Diagnostics: ", ina238_rcmp.v32, true);
}

//static void i2c_handler(char *cmd, char *response, uint16_t maxlen, uint32_t cs) {
//
//	char buf[30];
//	response[0] = 0;
//	for (uint8_t i = 0; i < 128; i++) {
//
//		if (HAL_I2C_IsDeviceReady(&hi2c1, (uint16_t)(i<<1), 3, 5) == HAL_OK) {
//		  // We got an ack
//			snprintf(buf, sizeof(buf), "%2x ", i);
//			strcat(response, buf);
//		} else {
//			strcat(response, "-- ");
//		}
//
//		if (i > 0 && (i + 1) % 16 == 0)
//			strcat(response, "\n");
//	}
//}
