#include <stdbool.h>
#include "main.h"
#include "pid.h"

/*Constructor (...)*********************************************************
 *    The parameters specified here are those for for which we can't set up
 *    reliable defaults, so we need to have the user set them.
 ***************************************************************************/
static void _pid_initialize(pid_t *instance) {
    pid_set_output_limits(instance, 0.0, 100.0);
    pid_set_tunings(instance, instance->kp, instance->ki, instance->kd, instance->p_on);
    instance->last_time = uwTick - instance->sample_time;
    instance->output_sum = *instance->output;
    instance->last_input = *instance->input;
    if (instance->output_sum > instance->out_max)
        instance->output_sum = instance->out_max;
    if (instance->output_sum < instance->out_min)
        instance->output_sum = instance->out_min;
}

void pid_initialize(pid_t *instance, double *input, double *output,
        double *setpoint, double p, double i, double d, int p_on,
        int controller_direction) {
    instance->output = output;
    instance->input = input;
    instance->setpoint = setpoint;
    instance->in_auto = false;
    instance->sample_time = 100;
    instance->controller_direction = controller_direction;

    pid_set_output_limits(instance, 0.0, 100.0);
    pid_set_tunings(instance, p, i, d, p_on);
    instance->last_time = uwTick - instance->sample_time;
    _pid_initialize(instance);
}

bool pid_compute(pid_t *instance) {
    if (!instance->in_auto)
        return false;

    unsigned long now = uwTick;

    uint32_t time_change = (now - instance->last_time);
    if (time_change >= instance->sample_time) {
        /*Compute all the working error variables*/
        double input = *instance->input;
        double error = *instance->setpoint - input;
        double dInput = (input - instance->last_input);
        instance->output_sum += (instance->ki * error);

        /*Add Proportional on Measurement, if P_ON_M is specified*/
        if (!instance->p_on_e)
            instance->output_sum -= instance->kp * dInput;

        if (instance->output_sum > instance->out_max)
            instance->output_sum = instance->out_max;

        if (instance->output_sum < instance->out_min)
            instance->output_sum = instance->out_min;

        /*Add Proportional on Error, if P_ON_E is specified*/
        double output;
        if (instance->p_on_e)
            output = instance->kp * error;
        else
            output = 0;

        /*Compute Rest of PID Output*/
        output += instance->output_sum - instance->kd * dInput;

        if (output > instance->out_max)
            output = instance->out_max;
        else if (output < instance->out_min)
            output = instance->out_min;
        *instance->output = output;

        /*Remember some variables for next time*/
        instance->last_input = input;
        instance->last_time = now;
        return true;
    } else
        return false;
}

/* SetTunings(...)*************************************************************
 * This function allows the controller's dynamic performance to be adjusted.
 * it's called automatically from the constructor, but tunings can also
 * be adjusted on the fly during normal operation
 ******************************************************************************/
void pid_set_tunings(pid_t *instance, double kp, double ki, double kd, int p_on) {
    if (kp < 0 || ki < 0 || kd < 0)
        return;

    instance->p_on = p_on;
    instance->p_on_e = p_on == PID_P_ON_E;

    double sample_time_in_sec = ((double) instance->sample_time) / 1000;
    ki = ki * sample_time_in_sec;
    kd = kd / sample_time_in_sec;

    if (instance->controller_direction == PID_REVERSE) {
        kp = (0 - kp);
        ki = (0 - ki);
        kd = (0 - kd);
    }
    instance->kp = kp;
    instance->ki = ki;
    instance->kd = kd;
}

void pid_set_output_limits(pid_t *instance, double min, double max) {
    if (min >= max)
        return;
    instance->out_min = min;
    instance->out_max = max;

    if (instance->in_auto) {
        if (*instance->output > instance->out_max)
            *instance->output = instance->out_max;

        if (*instance->output < instance->out_min)
            *instance->output = instance->out_min;

        if (instance->output_sum > instance->out_max)
            instance->output_sum = instance->out_max;
        if (instance->output_sum < instance->out_min)
            instance->output_sum = instance->out_min;
    }
}

/* SetMode(...)****************************************************************
 * Allows the controller Mode to be set to manual (0) or Automatic (non-zero)
 * when the transition from manual to auto occurs, the controller is
 * automatically initialized
 ******************************************************************************/
void pid_set_mode(pid_t *instance, int mode) {
    bool new_auto = (mode == PID_AUTO);
    if (new_auto && !instance->in_auto) { /*we just went from manual to auto*/
        _pid_initialize(instance);
    }
    instance->in_auto = new_auto;
}
